"""
URL configuration for taskmanager project.
"""
from django.contrib import admin
from django.urls import path, include
from django.conf import settings
from django.conf.urls.static import static
from django.contrib.auth import views as auth_views
# Import LoginView directly from users (using lazy loading in __init__.py)
from users import LoginView
# Importing users.urls which contains all the user and role views
from django.shortcuts import redirect, render
from django.http import (
    HttpResponseServerError, 
    HttpResponseNotFound,
    HttpResponseForbidden,
    HttpResponseBadRequest
)
from django.conf import settings
from . import views
from .views_dashboards import dashboard_router

# Error handlers
def custom_error_view(request, template_name, status):
    """Helper function to render error templates with proper context."""
    response = render(
        request, 
        template_name,
        {'request_path': request.path},
        status=status
    )
    return response

def handler400(request, exception=None, template_name='400.html'):
    return custom_error_view(request, template_name, 400)

def handler403(request, exception=None, template_name='403.html'):
    return custom_error_view(request, template_name, 403)

def handler404(request, exception=None, template_name='404.html'):
    return custom_error_view(request, template_name, 404)

def handler500(request, template_name='500.html'):
    return custom_error_view(request, template_name, 500)

urlpatterns = [
    # Session management
    path('keep-alive/', views.keep_alive, name='keep_alive'),
    
    # Homepage and Dashboard
    path('', views.home, name='home'),
    path('admin-dashboard/', views.admin_dashboard, name='admin_dashboard'),
    path('dashboard/', dashboard_router, name='dashboard'),
    
    # Include users app with its own URLs and namespace
    path('users/', include('users.urls', namespace='users')),
    
    # User authentication
    path('login/', LoginView.as_view(), name='login'),
    path('logout/', auth_views.LogoutView.as_view(next_page='home'), name='logout'),
    path('password_change/', auth_views.PasswordChangeView.as_view(template_name='registration/password_change.html'), 
         name='password_change'),
    path('password_change/done/', auth_views.PasswordChangeDoneView.as_view(template_name='registration/password_change_done.html'), 
         name='password_change_done'),
    path('password_reset/', auth_views.PasswordResetView.as_view(template_name='registration/password_reset.html'), 
         name='password_reset'),
    path('password_reset/done/', auth_views.PasswordResetDoneView.as_view(template_name='registration/password_reset_done.html'), 
         name='password_reset_done'),
    path('reset/<uidb64>/<token>/', auth_views.PasswordResetConfirmView.as_view(template_name='registration/password_reset_confirm.html'), 
         name='password_reset_confirm'),
    path('reset/done/', auth_views.PasswordResetCompleteView.as_view(template_name='registration/password_reset_complete.html'), 
         name='password_reset_complete'),
    
    # User profile
    path('profile/', views.profile, name='profile'),
    
    # Include other apps URLs
    path('tasks/', include('tasks.urls', namespace='tasks')),
    path('departments/', include(('users.department_urls', 'departments'), namespace='departments')),
    path('appointments/', include('appointments.urls')),
    path('reports/', include('reports.urls', namespace='reports')),
    
    # Admin site - Moved to bottom to prevent URL conflicts
    path('admin/', lambda request: redirect('admin_dashboard')),
]

# Debug toolbar
if settings.DEBUG:
    import debug_toolbar
    urlpatterns = [
        path('__debug__/', include(debug_toolbar.urls)),
    ] + urlpatterns
    
    # Serve media files in development
    urlpatterns += static(settings.MEDIA_URL, document_root=settings.MEDIA_ROOT)

# Custom error handlers
handler400 = 'taskmanager.views.handler400'
handler403 = 'taskmanager.views.handler403'
handler404 = 'taskmanager.views.handler404'
handler500 = 'taskmanager.views.handler500'
